/*
 * Copyright 2006 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.user.client.ui;

import java.util.EventListener;

/**
 * Event listener interface for keyboard events.
 */
public interface KeyboardListener extends EventListener {

  public static final int KEY_ALT = 18;
  public static final int KEY_BACKSPACE = 8;
  public static final int KEY_CTRL = 17;
  public static final int KEY_DELETE = 46;
  public static final int KEY_DOWN = 40;
  public static final int KEY_END = 35;
  public static final int KEY_ENTER = 13;
  public static final int KEY_ESCAPE = 27;
  public static final int KEY_HOME = 36;
  public static final int KEY_LEFT = 37;
  public static final int KEY_PAGEDOWN = 34;
  public static final int KEY_PAGEUP = 33;
  public static final int KEY_RIGHT = 39;
  public static final int KEY_SHIFT = 16;
  public static final int KEY_TAB = 9;
  public static final int KEY_UP = 38;

  public static final int MODIFIER_ALT = 4;
  public static final int MODIFIER_CTRL = 2;
  public static final int MODIFIER_SHIFT = 1;

  /**
   * Fired when the user depresses a physical key.
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the physical key that was depressed. Constants for this
   *          value are defined in this interface with the KEYCODE prefix.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyDown(Widget sender, char keyCode, int modifiers);

  /**
   * Fired when a keyboard action generates a character. This occurs after
   * onKeyDown and onKeyUp are fired for the physical key that was pressed.
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the Unicode character that was generated by the keyboard
   *          action.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyPress(Widget sender, char keyCode, int modifiers);

  /**
   * Fired when the user releases a physical key.
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the physical key that was released. Constants for this value
   *          are defined in this interface with the KEYCODE prefix.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyUp(Widget sender, char keyCode, int modifiers);
}
