/*
 * Decompiled with CFR 0.152.
 */
package org.dataone.hashstore.filehashstore;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dataone.hashstore.HashStore;
import org.dataone.hashstore.ObjectMetadata;
import org.dataone.hashstore.exceptions.CidNotFoundInPidRefsFileException;
import org.dataone.hashstore.exceptions.HashStoreRefsAlreadyExistException;
import org.dataone.hashstore.exceptions.IdentifierNotLockedException;
import org.dataone.hashstore.exceptions.MissingHexDigestsException;
import org.dataone.hashstore.exceptions.NonMatchingChecksumException;
import org.dataone.hashstore.exceptions.NonMatchingObjSizeException;
import org.dataone.hashstore.exceptions.OrphanPidRefsFileException;
import org.dataone.hashstore.exceptions.OrphanRefsFilesException;
import org.dataone.hashstore.exceptions.PidNotFoundInCidRefsFileException;
import org.dataone.hashstore.exceptions.PidRefsFileExistsException;
import org.dataone.hashstore.exceptions.PidRefsFileNotFoundException;
import org.dataone.hashstore.exceptions.UnsupportedHashAlgorithmException;
import org.dataone.hashstore.filehashstore.FileHashStoreUtility;

public class FileHashStore
implements HashStore {
    private static final Log logFileHashStore = LogFactory.getLog(FileHashStore.class);
    private static final int TIME_OUT_MILLISEC = 1000;
    private static final Collection<String> objectLockedCids = new ArrayList<String>(100);
    private static final Collection<String> objectLockedPids = new ArrayList<String>(100);
    private static final Collection<String> metadataLockedDocIds = new ArrayList<String>(100);
    private static final Collection<String> referenceLockedPids = new ArrayList<String>(100);
    private final Path STORE_ROOT;
    private final int DIRECTORY_DEPTH;
    private final int DIRECTORY_WIDTH;
    private final String OBJECT_STORE_ALGORITHM;
    private final Path OBJECT_STORE_DIRECTORY;
    private final Path OBJECT_TMP_FILE_DIRECTORY;
    private final String DEFAULT_METADATA_NAMESPACE;
    private final Path METADATA_STORE_DIRECTORY;
    private final Path METADATA_TMP_FILE_DIRECTORY;
    private final Path REFS_STORE_DIRECTORY;
    private final Path REFS_TMP_FILE_DIRECTORY;
    private final Path REFS_PID_FILE_DIRECTORY;
    private final Path REFS_CID_FILE_DIRECTORY;
    public static final String HASHSTORE_YAML = "hashstore.yaml";
    public static final String[] SUPPORTED_HASH_ALGORITHMS = new String[]{"MD2", "MD5", "SHA-1", "SHA-256", "SHA-384", "SHA-512", "SHA-512/224", "SHA-512/256"};

    public FileHashStore(Properties hashstoreProperties) throws IllegalArgumentException, IOException, NoSuchAlgorithmException {
        logFileHashStore.info((Object)"Initializing FileHashStore");
        FileHashStoreUtility.ensureNotNull(hashstoreProperties, "hashstoreProperties");
        Path storePath = Paths.get(hashstoreProperties.getProperty(HashStoreProperties.storePath.name()), new String[0]);
        int storeDepth = Integer.parseInt(hashstoreProperties.getProperty(HashStoreProperties.storeDepth.name()));
        int storeWidth = Integer.parseInt(hashstoreProperties.getProperty(HashStoreProperties.storeWidth.name()));
        String storeAlgorithm = hashstoreProperties.getProperty(HashStoreProperties.storeAlgorithm.name());
        String storeMetadataNamespace = hashstoreProperties.getProperty(HashStoreProperties.storeMetadataNamespace.name());
        this.verifyHashStoreProperties(storePath, storeDepth, storeWidth, storeAlgorithm, storeMetadataNamespace);
        this.STORE_ROOT = storePath;
        this.DIRECTORY_DEPTH = storeDepth;
        this.DIRECTORY_WIDTH = storeWidth;
        this.OBJECT_STORE_ALGORITHM = storeAlgorithm;
        this.DEFAULT_METADATA_NAMESPACE = storeMetadataNamespace;
        this.OBJECT_STORE_DIRECTORY = storePath.resolve("objects");
        this.METADATA_STORE_DIRECTORY = storePath.resolve("metadata");
        this.REFS_STORE_DIRECTORY = storePath.resolve("refs");
        this.OBJECT_TMP_FILE_DIRECTORY = this.OBJECT_STORE_DIRECTORY.resolve("tmp");
        this.METADATA_TMP_FILE_DIRECTORY = this.METADATA_STORE_DIRECTORY.resolve("tmp");
        this.REFS_TMP_FILE_DIRECTORY = this.REFS_STORE_DIRECTORY.resolve("tmp");
        this.REFS_PID_FILE_DIRECTORY = this.REFS_STORE_DIRECTORY.resolve("pids");
        this.REFS_CID_FILE_DIRECTORY = this.REFS_STORE_DIRECTORY.resolve("cids");
        try {
            Files.createDirectories(this.OBJECT_STORE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.METADATA_STORE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.REFS_STORE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.OBJECT_TMP_FILE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.METADATA_TMP_FILE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.REFS_TMP_FILE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.REFS_PID_FILE_DIRECTORY, new FileAttribute[0]);
            Files.createDirectories(this.REFS_CID_FILE_DIRECTORY, new FileAttribute[0]);
            logFileHashStore.debug((Object)"FileHashStore initialized");
        }
        catch (IOException ioe) {
            logFileHashStore.fatal((Object)("Failed to initialize FileHashStore - unable to create directories. Exception: " + ioe.getMessage()));
            throw ioe;
        }
        logFileHashStore.debug((Object)("HashStore initialized. Store Depth: " + this.DIRECTORY_DEPTH + ". Store Width: " + this.DIRECTORY_WIDTH + ". Store Algorithm: " + this.OBJECT_STORE_ALGORITHM + ". Store Metadata Namespace: " + this.DEFAULT_METADATA_NAMESPACE));
        Path hashstoreYaml = this.STORE_ROOT.resolve(HASHSTORE_YAML);
        if (!Files.exists(hashstoreYaml, new LinkOption[0])) {
            String hashstoreYamlContent = this.buildHashStoreYamlString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, this.OBJECT_STORE_ALGORITHM, this.DEFAULT_METADATA_NAMESPACE);
            this.writeHashStoreYaml(hashstoreYamlContent);
            logFileHashStore.info((Object)("hashstore.yaml written to storePath: " + hashstoreYaml));
        } else {
            logFileHashStore.info((Object)"hashstore.yaml exists and has been verified. Initializing FileHashStore.");
        }
    }

    protected void verifyHashStoreProperties(Path storePath, int storeDepth, int storeWidth, String storeAlgorithm, String storeMetadataNamespace) throws NoSuchAlgorithmException, IOException, IllegalArgumentException, IllegalStateException {
        if (storeDepth <= 0 || storeWidth <= 0) {
            String errMsg = "Depth and width must be > than 0. Depth: " + storeDepth + ". Width: " + storeWidth;
            logFileHashStore.fatal((Object)errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        this.validateAlgorithm(storeAlgorithm);
        FileHashStoreUtility.ensureNotNull(storeMetadataNamespace, "storeMetadataNamespace");
        FileHashStoreUtility.checkForNotEmptyAndValidString(storeMetadataNamespace, "storeMetadataNamespace");
        Path hashstoreYamlPredictedPath = Paths.get(storePath + "/hashstore.yaml", new String[0]);
        if (Files.exists(hashstoreYamlPredictedPath, new LinkOption[0])) {
            logFileHashStore.debug((Object)"hashstore.yaml found, checking properties.");
            HashMap<String, Object> hsProperties = this.loadHashStoreYaml(storePath);
            int existingStoreDepth = (Integer)hsProperties.get(HashStoreProperties.storeDepth.name());
            int existingStoreWidth = (Integer)hsProperties.get(HashStoreProperties.storeWidth.name());
            String existingStoreAlgorithm = (String)hsProperties.get(HashStoreProperties.storeAlgorithm.name());
            String existingStoreMetadataNs = (String)hsProperties.get(HashStoreProperties.storeMetadataNamespace.name());
            FileHashStoreUtility.checkObjectEquality("store depth", storeDepth, existingStoreDepth);
            FileHashStoreUtility.checkObjectEquality("store width", storeWidth, existingStoreWidth);
            FileHashStoreUtility.checkObjectEquality("store algorithm", storeAlgorithm, existingStoreAlgorithm);
            FileHashStoreUtility.checkObjectEquality("store metadata namespace", storeMetadataNamespace, existingStoreMetadataNs);
            logFileHashStore.info((Object)"hashstore.yaml found and HashStore verified");
        } else {
            logFileHashStore.debug((Object)"hashstore.yaml not found, checking store path for `/objects`, `/metadata` and `/refs` directories.");
            if (Files.isDirectory(storePath, new LinkOption[0])) {
                Path[] conflictingDirectories;
                for (Path dir : conflictingDirectories = new Path[]{storePath.resolve("objects"), storePath.resolve("metadata"), storePath.resolve("refs")}) {
                    if (!Files.exists(dir, new LinkOption[0]) || !Files.isDirectory(dir, new LinkOption[0])) continue;
                    String errMsg = "FileHashStore - Unable to initialize HashStore.`hashstore.yaml` is not found but potential conflicting directory exists: " + dir + ". Please choose a new folder or delete the conflicting directory and try again.";
                    logFileHashStore.fatal((Object)errMsg);
                    throw new IllegalStateException(errMsg);
                }
            }
            logFileHashStore.debug((Object)"hashstore.yaml not found. Supplied properties accepted.");
        }
    }

    protected HashMap<String, Object> loadHashStoreYaml(Path storePath) throws IOException {
        Path hashStoreYamlPath = storePath.resolve(HASHSTORE_YAML);
        File hashStoreYamlFile = hashStoreYamlPath.toFile();
        ObjectMapper om = new ObjectMapper((JsonFactory)new YAMLFactory());
        HashMap<String, Object> hsProperties = new HashMap<String, Object>();
        try {
            HashMap hashStoreYamlProperties = (HashMap)om.readValue(hashStoreYamlFile, HashMap.class);
            hsProperties.put(HashStoreProperties.storeDepth.name(), hashStoreYamlProperties.get("store_depth"));
            hsProperties.put(HashStoreProperties.storeWidth.name(), hashStoreYamlProperties.get("store_width"));
            hsProperties.put(HashStoreProperties.storeAlgorithm.name(), hashStoreYamlProperties.get("store_algorithm"));
            hsProperties.put(HashStoreProperties.storeMetadataNamespace.name(), hashStoreYamlProperties.get("store_metadata_namespace"));
        }
        catch (IOException ioe) {
            logFileHashStore.fatal((Object)(" Unable to retrieve 'hashstore.yaml'. IOException: " + ioe.getMessage()));
            throw ioe;
        }
        return hsProperties;
    }

    protected void writeHashStoreYaml(String yamlString) throws IOException {
        Path hashstoreYaml = this.STORE_ROOT.resolve(HASHSTORE_YAML);
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(Files.newOutputStream(hashstoreYaml, new OpenOption[0]), StandardCharsets.UTF_8));){
            writer.write(yamlString);
        }
        catch (IOException ioe) {
            logFileHashStore.fatal((Object)("Unable to write 'hashstore.yaml'. IOException: " + ioe.getMessage()));
            throw ioe;
        }
    }

    protected String buildHashStoreYamlString(int storeDepth, int storeWidth, String storeAlgorithm, String storeMetadataNamespace) {
        return String.format("# Default configuration variables for HashStore\n\n############### Directory Structure ###############\n# Desired amount of directories when sharding an object to form the permanent address\nstore_depth: %d  # WARNING: DO NOT CHANGE UNLESS SETTING UP NEW HASHSTORE\n# Width of directories created when sharding an object to form the permanent address\nstore_width: %d  # WARNING: DO NOT CHANGE UNLESS SETTING UP NEW HASHSTORE\n# Example:\n# Below, objects are shown listed in directories that are # levels deep (DIR_DEPTH=3),\n# with each directory consisting of 2 characters (DIR_WIDTH=2).\n#    /var/filehashstore/objects\n#    \u251c\u2500\u2500 7f\n#    \u2502   \u2514\u2500\u2500 5c\n#    \u2502       \u2514\u2500\u2500 c1\n#    \u2502           \u2514\u2500\u2500 8f0b04e812a3b4c8f686ce34e6fec558804bf61e54b176742a7f6368d6\n\n############### Format of the Metadata ###############\nstore_metadata_namespace: \"%s\"\n############### Hash Algorithms ###############\n# Hash algorithm to use when calculating object's hex digest for the permanent address\nstore_algorithm: \"%s\"\n############### Hash Algorithms ###############\n# Hash algorithm to use when calculating object's hex digest for the permanent address\n# The default algorithm list includes the hash algorithms calculated when storing an\n# object to disk and returned to the caller after successful storage.\nstore_default_algo_list:\n- \"MD5\"\n- \"SHA-1\"\n- \"SHA-256\"\n- \"SHA-384\"\n- \"SHA-512\"\n", storeDepth, storeWidth, storeMetadataNamespace, storeAlgorithm);
    }

    @Override
    public ObjectMetadata storeObject(InputStream object, String pid, String additionalAlgorithm, String checksum, String checksumAlgorithm, long objSize) throws NoSuchAlgorithmException, IOException, RuntimeException, InterruptedException {
        logFileHashStore.debug((Object)("Storing data object for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(object, "object");
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        if (additionalAlgorithm != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(additionalAlgorithm, "additionalAlgorithm");
            this.validateAlgorithm(additionalAlgorithm);
        }
        if (checksumAlgorithm != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksumAlgorithm, "checksumAlgorithm");
            this.validateAlgorithm(checksumAlgorithm);
        }
        if (objSize != -1L) {
            FileHashStoreUtility.checkPositive(objSize);
        }
        try (InputStream inputStream = object;){
            ObjectMetadata objectMetadata = this.syncPutObject(object, pid, additionalAlgorithm, checksum, checksumAlgorithm, objSize);
            return objectMetadata;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ObjectMetadata syncPutObject(InputStream object, String pid, String additionalAlgorithm, String checksum, String checksumAlgorithm, long objSize) throws NoSuchAlgorithmException, IOException, RuntimeException, InterruptedException {
        try {
            Collection<String> collection = objectLockedPids;
            synchronized (collection) {
                if (objectLockedPids.contains(pid)) {
                    String errMsg = "Duplicate object request encountered for pid: " + pid + ". Already in progress.";
                    logFileHashStore.warn((Object)errMsg);
                    throw new RuntimeException(errMsg);
                }
                logFileHashStore.debug((Object)("Synchronizing objectLockedPids for pid: " + pid));
                objectLockedPids.add(pid);
            }
            logFileHashStore.debug((Object)("putObject() called to store pid: " + pid + ". additionalAlgorithm: " + additionalAlgorithm + ". checksum: " + checksum + ". checksumAlgorithm: " + checksumAlgorithm));
            ObjectMetadata objInfo = this.putObject(object, pid, additionalAlgorithm, checksum, checksumAlgorithm, objSize);
            String cid = objInfo.cid();
            this.tagObject(pid, cid);
            logFileHashStore.info((Object)("Object stored for pid: " + pid + " at " + this.getHashStoreDataObjectPath(pid)));
            ObjectMetadata objectMetadata = objInfo;
            return objectMetadata;
        }
        catch (NoSuchAlgorithmException nsae) {
            String errMsg = "Unable to store object for pid: " + pid + ". NoSuchAlgorithmException: " + nsae.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw nsae;
        }
        catch (PidRefsFileExistsException prfee) {
            String errMsg = "Unable to store object for pid: " + pid + ". PidRefsFileExistsException: " + prfee.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw prfee;
        }
        catch (IOException ioe) {
            String errMsg = "Unable to store object for pid: " + pid + ". IOException: " + ioe.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw ioe;
        }
        catch (RuntimeException re) {
            String errMsg = "Unable to store object for pid: " + pid + ". Runtime Exception: " + re.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw re;
        }
        finally {
            FileHashStore.releaseObjectLockedPids(pid);
        }
    }

    @Override
    public ObjectMetadata storeObject(InputStream object) throws NoSuchAlgorithmException, IOException, RuntimeException, InterruptedException {
        try (InputStream inputStream = object;){
            ObjectMetadata objectMetadata = this.putObject(object, "HashStoreNoPid", null, null, null, -1L);
            return objectMetadata;
        }
    }

    @Override
    public void tagObject(String pid, String cid) throws IOException, NoSuchAlgorithmException, InterruptedException {
        logFileHashStore.debug((Object)("Tagging cid (" + cid + ") with pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.ensureNotNull(cid, "cid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(cid, "cid");
        try {
            this.storeHashStoreRefsFiles(pid, cid);
        }
        catch (HashStoreRefsAlreadyExistException hsrfae) {
            String errMsg = "HashStore refs files already exist for pid " + pid + " and cid: " + cid;
            throw new HashStoreRefsAlreadyExistException(errMsg);
        }
        catch (PidRefsFileExistsException prfe) {
            String errMsg = "pid: " + pid + " already references another cid. A pid can only reference one cid.";
            throw new PidRefsFileExistsException(errMsg);
        }
    }

    @Override
    public String storeMetadata(InputStream metadata, String pid, String formatId) throws IOException, IllegalArgumentException, InterruptedException, NoSuchAlgorithmException {
        String checkedFormatId;
        logFileHashStore.debug((Object)("Storing metadata for pid: " + pid + ", with formatId: " + formatId));
        FileHashStoreUtility.ensureNotNull(metadata, "metadata");
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        if (formatId == null) {
            checkedFormatId = this.DEFAULT_METADATA_NAMESPACE;
        } else {
            FileHashStoreUtility.checkForNotEmptyAndValidString(formatId, "formatId");
            checkedFormatId = formatId;
        }
        try (InputStream inputStream = metadata;){
            String string = this.syncPutMetadata(metadata, pid, checkedFormatId);
            return string;
        }
    }

    private String syncPutMetadata(InputStream metadata, String pid, String checkedFormatId) throws InterruptedException, IOException, NoSuchAlgorithmException {
        String pidFormatId = pid + checkedFormatId;
        String metadataDocId = FileHashStoreUtility.getPidHexDigest(pidFormatId, this.OBJECT_STORE_ALGORITHM);
        logFileHashStore.debug((Object)("putMetadata() called to store metadata for pid: " + pid + ", with formatId: " + checkedFormatId + " for metadata document: " + metadataDocId));
        try {
            FileHashStore.synchronizeMetadataLockedDocIds(metadataDocId);
            String pathToStoredMetadata = this.putMetadata(metadata, pid, checkedFormatId);
            logFileHashStore.info((Object)("Metadata stored for pid: " + pid + " at: " + pathToStoredMetadata));
            String string = pathToStoredMetadata;
            return string;
        }
        catch (IOException ioe) {
            String errMsg = "Unable to store metadata, IOException encountered: " + ioe.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw ioe;
        }
        catch (NoSuchAlgorithmException nsae) {
            String errMsg = "Unable to store metadata, algorithm to calculate permanent address is not supported: " + nsae.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw nsae;
        }
        finally {
            FileHashStore.releaseMetadataLockedDocIds(metadataDocId);
        }
    }

    @Override
    public String storeMetadata(InputStream metadata, String pid) throws IOException, IllegalArgumentException, InterruptedException, NoSuchAlgorithmException {
        return this.storeMetadata(metadata, pid, this.DEFAULT_METADATA_NAMESPACE);
    }

    @Override
    public InputStream retrieveObject(String pid) throws IllegalArgumentException, IOException, NoSuchAlgorithmException {
        logFileHashStore.debug((Object)("Retrieving InputStream to data object for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        Path objRealPath = this.getHashStoreDataObjectPath(pid);
        if (!Files.exists(objRealPath, new LinkOption[0])) {
            String errMsg = "File does not exist for pid: " + pid + " with object address: " + objRealPath;
            logFileHashStore.warn((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        try {
            InputStream objectCidInputStream = Files.newInputStream(objRealPath, new OpenOption[0]);
            logFileHashStore.info((Object)("Retrieved object for pid: " + pid));
            return objectCidInputStream;
        }
        catch (IOException ioe) {
            String errMsg = "Unexpected error when creating InputStream for pid: " + pid + ", IOException: " + ioe.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw new IOException(errMsg);
        }
    }

    @Override
    public InputStream retrieveMetadata(String pid, String formatId) throws IllegalArgumentException, IOException, NoSuchAlgorithmException {
        logFileHashStore.debug((Object)("Retrieving metadata document for pid: " + pid + " with formatId: " + formatId));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        FileHashStoreUtility.ensureNotNull(formatId, "formatId");
        FileHashStoreUtility.checkForNotEmptyAndValidString(formatId, "formatId");
        return this.getHashStoreMetadataInputStream(pid, formatId);
    }

    @Override
    public InputStream retrieveMetadata(String pid) throws IllegalArgumentException, IOException, NoSuchAlgorithmException {
        logFileHashStore.debug((Object)("Retrieving metadata for pid: " + pid + " with default metadata namespace: "));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        return this.getHashStoreMetadataInputStream(pid, this.DEFAULT_METADATA_NAMESPACE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void deleteObject(String pid) throws IllegalArgumentException, IOException, NoSuchAlgorithmException, InterruptedException {
        logFileHashStore.debug((Object)("Deleting object for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "id");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "id");
        ArrayList<Path> deleteList = new ArrayList<Path>();
        try {
            FileHashStore.synchronizeObjectLockedPids(pid);
            try {
                ObjectInfo objInfoMap = this.findObject(pid);
                String cid = objInfoMap.cid();
                FileHashStore.synchronizeObjectLockedCids(cid);
                try {
                    Path absCidRefsPath = this.getHashStoreRefsPath(cid, HashStoreIdTypes.cid);
                    Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
                    this.updateRefsFile(pid, absCidRefsPath, HashStoreRefUpdateTypes.remove);
                    if (Files.size(absCidRefsPath) == 0L) {
                        Path objRealPath = this.getHashStoreDataObjectPath(pid);
                        deleteList.add(FileHashStoreUtility.renamePathForDeletion(objRealPath));
                        deleteList.add(FileHashStoreUtility.renamePathForDeletion(absCidRefsPath));
                    } else {
                        String warnMsg = "cid referenced by pid: " + pid + " is not empty (refs exist for cid). Skipping object deletion.";
                        logFileHashStore.warn((Object)warnMsg);
                    }
                    deleteList.add(FileHashStoreUtility.renamePathForDeletion(absPidRefsPath));
                    FileHashStoreUtility.deleteListItems(deleteList);
                    this.deleteMetadata(pid);
                    logFileHashStore.info((Object)("Data file and references deleted for: " + pid));
                }
                finally {
                    FileHashStore.releaseObjectLockedCids(cid);
                }
            }
            catch (OrphanPidRefsFileException oprfe) {
                Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
                deleteList.add(FileHashStoreUtility.renamePathForDeletion(absPidRefsPath));
                FileHashStoreUtility.deleteListItems(deleteList);
                this.deleteMetadata(pid);
                String warnMsg = "Cid refs file does not exist for pid: " + pid + ". Deleted orphan pid refs file and metadata.";
                logFileHashStore.warn((Object)warnMsg);
            }
            catch (OrphanRefsFilesException orfe) {
                Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
                String cidRead = new String(Files.readAllBytes(absPidRefsPath));
                try {
                    FileHashStore.synchronizeObjectLockedCids(cidRead);
                    Path absCidRefsPath = this.getHashStoreRefsPath(cidRead, HashStoreIdTypes.cid);
                    this.updateRefsFile(pid, absCidRefsPath, HashStoreRefUpdateTypes.remove);
                    if (Files.size(absCidRefsPath) == 0L) {
                        deleteList.add(FileHashStoreUtility.renamePathForDeletion(absCidRefsPath));
                    }
                    deleteList.add(FileHashStoreUtility.renamePathForDeletion(absPidRefsPath));
                    FileHashStoreUtility.deleteListItems(deleteList);
                    this.deleteMetadata(pid);
                    String warnMsg = "Object with cid: " + cidRead + " does not exist, but pid and cid reference file found for pid: " + pid + ". Deleted pid and cid ref files and metadata.";
                    logFileHashStore.warn((Object)warnMsg);
                }
                finally {
                    FileHashStore.releaseObjectLockedCids(cidRead);
                }
            }
            catch (PidNotFoundInCidRefsFileException pnficrfe) {
                Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
                deleteList.add(FileHashStoreUtility.renamePathForDeletion(absPidRefsPath));
                FileHashStoreUtility.deleteListItems(deleteList);
                this.deleteMetadata(pid);
                String warnMsg = "Pid not found in expected cid refs file for pid: " + pid + ". Deleted orphan pid refs file and metadata.";
                logFileHashStore.warn((Object)warnMsg);
            }
        }
        finally {
            FileHashStore.releaseObjectLockedPids(pid);
        }
    }

    @Override
    public void deleteIfInvalidObject(ObjectMetadata objectInfo, String checksum, String checksumAlgorithm, long objSize) throws NonMatchingObjSizeException, NonMatchingChecksumException, UnsupportedHashAlgorithmException, InterruptedException, NoSuchAlgorithmException, IOException {
        String errMsg;
        logFileHashStore.debug((Object)("Verifying data object for cid: " + objectInfo.cid()));
        FileHashStoreUtility.ensureNotNull(objectInfo, "objectInfo");
        FileHashStoreUtility.ensureNotNull(objectInfo.hexDigests(), "objectInfo.getHexDigests()");
        if (objectInfo.hexDigests().isEmpty()) {
            throw new MissingHexDigestsException("Missing hexDigests in supplied ObjectMetadata");
        }
        FileHashStoreUtility.ensureNotNull(checksum, "checksum");
        FileHashStoreUtility.ensureNotNull(checksumAlgorithm, "checksumAlgorithm");
        FileHashStoreUtility.checkPositive(objSize);
        String objCid = objectInfo.cid();
        long objInfoRetrievedSize = objectInfo.size();
        Map<String, String> hexDigests = objectInfo.hexDigests();
        String digestFromHexDigests = hexDigests.get(checksumAlgorithm);
        if (digestFromHexDigests == null) {
            try {
                this.validateAlgorithm(checksumAlgorithm);
                String objRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, objCid);
                Path pathToCidObject = this.OBJECT_STORE_DIRECTORY.resolve(objRelativePath);
                try (InputStream inputStream = Files.newInputStream(pathToCidObject, new OpenOption[0]);){
                    digestFromHexDigests = FileHashStoreUtility.calculateHexDigest(inputStream, checksumAlgorithm);
                }
                catch (IOException ioe) {
                    String errMsg2 = "Unexpected error when calculating a checksum for cid: " + objCid + " with algorithm (" + checksumAlgorithm + ") that is not part of the default list. " + ioe.getMessage();
                    throw new IOException(errMsg2);
                }
            }
            catch (NoSuchAlgorithmException nsae) {
                String errMsg3 = "checksumAlgorithm given: " + checksumAlgorithm + " is not supported. Supported algorithms: " + Arrays.toString(SUPPORTED_HASH_ALGORITHMS);
                logFileHashStore.error((Object)errMsg3);
                throw new UnsupportedHashAlgorithmException(errMsg3);
            }
        }
        if (!digestFromHexDigests.equals(checksum)) {
            this.deleteObjectByCid(objCid);
            errMsg = "Object content invalid for cid: " + objCid + ". Expected checksum: " + checksum + ". Actual checksum calculated: " + digestFromHexDigests + " (algorithm: " + checksumAlgorithm + ")";
            logFileHashStore.error((Object)errMsg);
            throw new NonMatchingChecksumException(errMsg, hexDigests);
        }
        if (objInfoRetrievedSize != objSize) {
            this.deleteObjectByCid(objCid);
            errMsg = "Object size invalid for cid: " + objCid + ". Expected size: " + objSize + ". Actual size: " + objInfoRetrievedSize;
            logFileHashStore.error((Object)errMsg);
            throw new NonMatchingObjSizeException(errMsg);
        }
        String infoMsg = "Object has been validated for cid: " + objCid + ". Expected checksum: " + checksum + ". Actual checksum calculated: " + digestFromHexDigests + " (algorithm: " + checksumAlgorithm + ")";
        logFileHashStore.info((Object)infoMsg);
    }

    @Override
    public void deleteMetadata(String pid, String formatId) throws IllegalArgumentException, IOException, NoSuchAlgorithmException, InterruptedException {
        logFileHashStore.debug((Object)("Deleting metadata document for pid: " + pid + " with formatId: " + formatId));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        FileHashStoreUtility.ensureNotNull(formatId, "formatId");
        FileHashStoreUtility.checkForNotEmptyAndValidString(formatId, "formatId");
        Path metadataDocPath = this.getHashStoreMetadataPath(pid, formatId);
        ArrayList<Path> metadataDocPaths = new ArrayList<Path>();
        metadataDocPaths.add(metadataDocPath);
        if (!metadataDocPaths.isEmpty()) {
            Collection<Path> deleteList = this.syncRenameMetadataDocForDeletion(metadataDocPaths);
            FileHashStoreUtility.deleteListItems(deleteList);
        }
        logFileHashStore.info((Object)("Metadata document deleted for: " + pid + " with metadata address: " + metadataDocPath));
    }

    @Override
    public void deleteMetadata(String pid) throws IllegalArgumentException, IOException, NoSuchAlgorithmException, InterruptedException {
        logFileHashStore.debug((Object)("Deleting all metadata documents for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        String pidHexDigest = FileHashStoreUtility.getPidHexDigest(pid, this.OBJECT_STORE_ALGORITHM);
        String pidRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, pidHexDigest);
        Path expectedPidMetadataDirectory = this.METADATA_STORE_DIRECTORY.resolve(pidRelativePath);
        List<Path> metadataDocPaths = FileHashStoreUtility.getFilesFromDir(expectedPidMetadataDirectory);
        if (!metadataDocPaths.isEmpty()) {
            Collection<Path> deleteList = this.syncRenameMetadataDocForDeletion(metadataDocPaths);
            FileHashStoreUtility.deleteListItems(deleteList);
        }
        logFileHashStore.info((Object)("All metadata documents deleted for: " + pid));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Collection<Path> syncRenameMetadataDocForDeletion(Collection<Path> metadataDocPaths) throws IOException, InterruptedException {
        FileHashStoreUtility.ensureNotNull(metadataDocPaths, "metadataDocPaths");
        if (metadataDocPaths.isEmpty()) {
            String errMsg = "metadataDocPaths supplied cannot be empty.";
            logFileHashStore.error((Object)errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        ArrayList<Path> metadataDocsToDelete = new ArrayList<Path>();
        try {
            for (Path metadataDocToDelete : metadataDocPaths) {
                String metadataDocId = metadataDocToDelete.getFileName().toString();
                try {
                    FileHashStore.synchronizeMetadataLockedDocIds(metadataDocId);
                    if (!Files.exists(metadataDocToDelete, new LinkOption[0])) continue;
                    metadataDocsToDelete.add(FileHashStoreUtility.renamePathForDeletion(metadataDocToDelete));
                }
                finally {
                    FileHashStore.releaseMetadataLockedDocIds(metadataDocId);
                }
            }
        }
        catch (Exception ge) {
            if (!metadataDocsToDelete.isEmpty()) {
                for (Path metadataDocToPlaceBack : metadataDocsToDelete) {
                    Path fileNameWithDeleted = metadataDocToPlaceBack.getFileName();
                    String metadataDocId = fileNameWithDeleted.toString().replace("_delete", "");
                    try {
                        FileHashStore.synchronizeMetadataLockedDocIds(metadataDocId);
                        if (!Files.exists(metadataDocToPlaceBack, new LinkOption[0])) continue;
                        FileHashStoreUtility.renamePathForRestoration(metadataDocToPlaceBack);
                    }
                    finally {
                        FileHashStore.releaseMetadataLockedDocIds(metadataDocId);
                    }
                }
            }
            String errMsg = "An unexpected exception has occurred when deleting metadata documents. Attempts to restore all affected metadata documents have been made. Additional details: " + ge.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw ge;
        }
        return metadataDocsToDelete;
    }

    @Override
    public String getHexDigest(String pid, String algorithm) throws IllegalArgumentException, IOException, NoSuchAlgorithmException {
        logFileHashStore.debug((Object)("Calculating hex digest for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        this.validateAlgorithm(algorithm);
        if (algorithm.equals(this.OBJECT_STORE_ALGORITHM)) {
            ObjectInfo objInfo = this.findObject(pid);
            return objInfo.cid();
        }
        Path objRealPath = this.getHashStoreDataObjectPath(pid);
        if (!Files.exists(objRealPath, new LinkOption[0])) {
            String errMsg = "File does not exist for pid: " + pid + " with object address: " + objRealPath;
            logFileHashStore.warn((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        InputStream dataStream = Files.newInputStream(objRealPath, new OpenOption[0]);
        String mdObjectHexDigest = FileHashStoreUtility.calculateHexDigest(dataStream, algorithm);
        logFileHashStore.info((Object)("Hex digest calculated for pid: " + pid + ", with hex digest value: " + mdObjectHexDigest));
        return mdObjectHexDigest;
    }

    protected ObjectInfo findObject(String pid) throws NoSuchAlgorithmException, IOException, OrphanPidRefsFileException, PidNotFoundInCidRefsFileException, OrphanRefsFilesException, PidRefsFileNotFoundException {
        logFileHashStore.debug((Object)("Finding object for pid: " + pid));
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
        if (Files.exists(absPidRefsPath, new LinkOption[0])) {
            String cid = new String(Files.readAllBytes(absPidRefsPath));
            Path absCidRefsPath = this.getHashStoreRefsPath(cid, HashStoreIdTypes.cid);
            if (!Files.exists(absCidRefsPath, new LinkOption[0])) {
                String errMsg = "Cid refs file does not exist for cid: " + cid + " with address: " + absCidRefsPath + ", but pid refs file exists.";
                logFileHashStore.error((Object)errMsg);
                throw new OrphanPidRefsFileException(errMsg);
            }
            if (this.isStringInRefsFile(pid, absCidRefsPath)) {
                logFileHashStore.info((Object)("cid (" + cid + ") found for pid: " + pid));
                String objRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, cid);
                Path realPath = this.OBJECT_STORE_DIRECTORY.resolve(objRelativePath);
                if (Files.exists(realPath, new LinkOption[0])) {
                    Path metadataPidExpectedPath = this.getHashStoreMetadataPath(pid, this.DEFAULT_METADATA_NAMESPACE);
                    if (Files.exists(metadataPidExpectedPath, new LinkOption[0])) {
                        return new ObjectInfo(cid, realPath.toString(), absCidRefsPath.toString(), absPidRefsPath.toString(), metadataPidExpectedPath.toString());
                    }
                    return new ObjectInfo(cid, realPath.toString(), absCidRefsPath.toString(), absPidRefsPath.toString(), "Does not exist");
                }
                String errMsg = "Object with cid: " + cid + " does not exist, but pid and cid reference file found for pid: " + pid;
                logFileHashStore.error((Object)errMsg);
                throw new OrphanRefsFilesException(errMsg);
            }
            String errMsg = "Pid refs file exists, but pid (" + pid + ") not found in cid refs file for cid: " + cid + " with address: " + absCidRefsPath;
            logFileHashStore.error((Object)errMsg);
            throw new PidNotFoundInCidRefsFileException(errMsg);
        }
        String errMsg = "Unable to find cid for pid: " + pid + ". Pid refs file does not exist at: " + absPidRefsPath;
        logFileHashStore.error((Object)errMsg);
        throw new PidRefsFileNotFoundException(errMsg);
    }

    protected ObjectMetadata putObject(InputStream object, String pid, String additionalAlgorithm, String checksum, String checksumAlgorithm, long objSize) throws IOException, NoSuchAlgorithmException, SecurityException, FileNotFoundException, PidRefsFileExistsException, IllegalArgumentException, NullPointerException, AtomicMoveNotSupportedException, InterruptedException {
        Map<String, String> hexDigests;
        logFileHashStore.debug((Object)("Begin writing data object for pid: " + pid));
        boolean compareChecksum = this.verifyChecksumParameters(checksum, checksumAlgorithm);
        if (additionalAlgorithm != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(additionalAlgorithm, "additionalAlgorithm");
            this.validateAlgorithm(additionalAlgorithm);
        }
        if (objSize != -1L) {
            FileHashStoreUtility.checkPositive(objSize);
        }
        File tmpFile = FileHashStoreUtility.generateTmpFile("tmp", this.OBJECT_TMP_FILE_DIRECTORY);
        try {
            hexDigests = this.writeToTmpFileAndGenerateChecksums(tmpFile, object, additionalAlgorithm, checksumAlgorithm);
        }
        catch (Exception ge) {
            Files.delete(tmpFile.toPath());
            String errMsg = "Unexpected Exception while storing object for pid: " + pid + ". " + ge.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw new IOException(errMsg);
        }
        this.validateTmpObject(compareChecksum, checksum, checksumAlgorithm, tmpFile, hexDigests, objSize);
        String objectCid = hexDigests.get(this.OBJECT_STORE_ALGORITHM);
        String objRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, objectCid);
        Path objRealPath = this.OBJECT_STORE_DIRECTORY.resolve(objRelativePath);
        try {
            FileHashStore.synchronizeObjectLockedCids(objectCid);
            if (!Files.exists(objRealPath, new LinkOption[0])) {
                logFileHashStore.info((Object)("Storing tmpFile: " + tmpFile));
                File permFile = objRealPath.toFile();
                this.move(tmpFile, permFile, "object");
                logFileHashStore.debug((Object)("Successfully moved data object: " + objRealPath));
            } else {
                Files.delete(tmpFile.toPath());
                String errMsg = "File already exists for pid: " + pid + ". Object address: " + objRealPath + ". Deleting temporary file: " + tmpFile;
                logFileHashStore.warn((Object)errMsg);
            }
        }
        catch (Exception e) {
            String errMsg = "Unexpected exception when moving object with cid: " + objectCid + " for pid:" + pid + ". Additional Details: " + e.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw e;
        }
        finally {
            FileHashStore.releaseObjectLockedCids(objectCid);
        }
        return new ObjectMetadata(pid, objectCid, Files.size(objRealPath), hexDigests);
    }

    protected void validateTmpObject(boolean compareChecksum, String checksum, String checksumAlgorithm, File tmpFile, Map<String, String> hexDigests, long expectedSize) throws NoSuchAlgorithmException, NonMatchingChecksumException, NonMatchingObjSizeException, IOException {
        long storedObjFileSize;
        if (expectedSize > 0L && expectedSize != (storedObjFileSize = Files.size(Paths.get(tmpFile.toString(), new String[0])))) {
            try {
                Files.delete(tmpFile.toPath());
            }
            catch (Exception ge) {
                String errMsg = "objSize given is not equal to the stored object size. ObjSize: " + expectedSize + ". storedObjFileSize: " + storedObjFileSize + ". Failed to delete tmpFile: " + tmpFile + ". " + ge.getMessage();
                logFileHashStore.error((Object)errMsg);
                throw new NonMatchingObjSizeException(errMsg);
            }
            String errMsg = "objSize given is not equal to the stored object size. ObjSize: " + expectedSize + ". storedObjFileSize: " + storedObjFileSize + ". Deleting tmpFile: " + tmpFile;
            logFileHashStore.error((Object)errMsg);
            throw new NonMatchingObjSizeException(errMsg);
        }
        if (compareChecksum) {
            FileHashStoreUtility.ensureNotNull(hexDigests, "hexDigests");
            logFileHashStore.info((Object)"Validating object, checksum arguments supplied and valid.");
            String digestFromHexDigests = hexDigests.get(checksumAlgorithm);
            if (digestFromHexDigests == null) {
                String baseErrMsg = "Object cannot be validated. Algorithm not found in given hexDigests map. Algorithm requested: " + checksumAlgorithm;
                try {
                    Files.delete(tmpFile.toPath());
                }
                catch (Exception ge) {
                    String errMsg = baseErrMsg + ". Failed to delete tmpFile: " + tmpFile + ". " + ge.getMessage();
                    logFileHashStore.error((Object)errMsg);
                    throw new NonMatchingChecksumException(errMsg, hexDigests);
                }
                String errMsg = baseErrMsg + ". tmpFile has been deleted: " + tmpFile;
                logFileHashStore.error((Object)errMsg);
                throw new NoSuchAlgorithmException(errMsg);
            }
            if (!checksum.equalsIgnoreCase(digestFromHexDigests)) {
                String baseErrMsg = "Checksum given is not equal to the calculated hex digest: " + digestFromHexDigests + ". Checksum provided: " + checksum;
                try {
                    Files.delete(tmpFile.toPath());
                }
                catch (Exception ge) {
                    String errMsg = baseErrMsg + ". Failed to delete tmpFile: " + tmpFile + ". " + ge.getMessage();
                    logFileHashStore.error((Object)errMsg);
                    throw new NonMatchingChecksumException(errMsg, hexDigests);
                }
                String errMsg = baseErrMsg + ". tmpFile has been deleted: " + tmpFile;
                logFileHashStore.error((Object)errMsg);
                throw new NonMatchingChecksumException(errMsg, hexDigests);
            }
        }
    }

    protected boolean validateAlgorithm(String algorithm) throws NullPointerException, IllegalArgumentException, NoSuchAlgorithmException {
        FileHashStoreUtility.ensureNotNull(algorithm, "algorithm");
        FileHashStoreUtility.checkForNotEmptyAndValidString(algorithm, "algorithm");
        boolean algorithmSupported = Arrays.asList(SUPPORTED_HASH_ALGORITHMS).contains(algorithm);
        if (!algorithmSupported) {
            String errMsg = "Algorithm not supported: " + algorithm + ". Supported algorithms: " + Arrays.toString(SUPPORTED_HASH_ALGORITHMS);
            logFileHashStore.error((Object)errMsg);
            throw new NoSuchAlgorithmException(errMsg);
        }
        return true;
    }

    protected boolean shouldCalculateAlgorithm(String algorithm) {
        FileHashStoreUtility.ensureNotNull(algorithm, "algorithm");
        FileHashStoreUtility.checkForNotEmptyAndValidString(algorithm, "algorithm");
        boolean shouldCalculateAlgorithm = true;
        for (DefaultHashAlgorithms defAlgo : DefaultHashAlgorithms.values()) {
            if (!algorithm.equals(defAlgo.getName())) continue;
            shouldCalculateAlgorithm = false;
            break;
        }
        return shouldCalculateAlgorithm;
    }

    protected boolean verifyChecksumParameters(String checksum, String checksumAlgorithm) throws NoSuchAlgorithmException {
        if (checksumAlgorithm != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksumAlgorithm, "checksumAlgorithm");
            this.validateAlgorithm(checksumAlgorithm);
        }
        if (checksum != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksum, "checksum");
        }
        if (checksum != null && !checksum.trim().isEmpty()) {
            FileHashStoreUtility.ensureNotNull(checksumAlgorithm, "checksumAlgorithm");
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksumAlgorithm, "algorithm");
        }
        boolean requestValidation = false;
        if (checksumAlgorithm != null && !checksumAlgorithm.trim().isEmpty() && (requestValidation = this.validateAlgorithm(checksumAlgorithm))) {
            FileHashStoreUtility.ensureNotNull(checksum, "checksum");
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksum, "checksum");
        }
        return requestValidation;
    }

    protected Map<String, String> writeToTmpFileAndGenerateChecksums(File tmpFile, InputStream dataStream, String additionalAlgorithm, String checksumAlgorithm) throws NoSuchAlgorithmException, IOException, FileNotFoundException, SecurityException {
        boolean generateAddAlgo = false;
        if (additionalAlgorithm != null) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(additionalAlgorithm, "additionalAlgorithm");
            this.validateAlgorithm(additionalAlgorithm);
            generateAddAlgo = this.shouldCalculateAlgorithm(additionalAlgorithm);
        }
        boolean generateCsAlgo = false;
        if (checksumAlgorithm != null && !checksumAlgorithm.equals(additionalAlgorithm)) {
            FileHashStoreUtility.checkForNotEmptyAndValidString(checksumAlgorithm, "checksumAlgorithm");
            this.validateAlgorithm(checksumAlgorithm);
            generateCsAlgo = this.shouldCalculateAlgorithm(checksumAlgorithm);
        }
        FileOutputStream os = new FileOutputStream(tmpFile);
        MessageDigest md5 = MessageDigest.getInstance(DefaultHashAlgorithms.MD5.getName());
        MessageDigest sha1 = MessageDigest.getInstance(DefaultHashAlgorithms.SHA_1.getName());
        MessageDigest sha256 = MessageDigest.getInstance(DefaultHashAlgorithms.SHA_256.getName());
        MessageDigest sha384 = MessageDigest.getInstance(DefaultHashAlgorithms.SHA_384.getName());
        MessageDigest sha512 = MessageDigest.getInstance(DefaultHashAlgorithms.SHA_512.getName());
        MessageDigest additionalAlgo = null;
        MessageDigest checksumAlgo = null;
        if (generateAddAlgo) {
            logFileHashStore.debug((Object)("Adding additional algorithm to hex digest map, algorithm: " + additionalAlgorithm));
            additionalAlgo = MessageDigest.getInstance(additionalAlgorithm);
        }
        if (generateCsAlgo) {
            logFileHashStore.debug((Object)("Adding checksum algorithm to hex digest map, algorithm: " + checksumAlgorithm));
            checksumAlgo = MessageDigest.getInstance(checksumAlgorithm);
        }
        try (InputStream inputStream = dataStream;){
            int bytesRead;
            byte[] buffer = new byte[8192];
            while ((bytesRead = dataStream.read(buffer)) != -1) {
                os.write(buffer, 0, bytesRead);
                md5.update(buffer, 0, bytesRead);
                sha1.update(buffer, 0, bytesRead);
                sha256.update(buffer, 0, bytesRead);
                sha384.update(buffer, 0, bytesRead);
                sha512.update(buffer, 0, bytesRead);
                if (generateAddAlgo) {
                    additionalAlgo.update(buffer, 0, bytesRead);
                }
                if (!generateCsAlgo) continue;
                checksumAlgo.update(buffer, 0, bytesRead);
            }
        }
        catch (IOException ioe) {
            String errMsg = "Unexpected Exception ~ " + ioe.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw ioe;
        }
        finally {
            os.flush();
            os.close();
        }
        HashMap<String, String> hexDigests = new HashMap<String, String>();
        String md5Digest = DatatypeConverter.printHexBinary((byte[])md5.digest()).toLowerCase();
        String sha1Digest = DatatypeConverter.printHexBinary((byte[])sha1.digest()).toLowerCase();
        String sha256Digest = DatatypeConverter.printHexBinary((byte[])sha256.digest()).toLowerCase();
        String sha384Digest = DatatypeConverter.printHexBinary((byte[])sha384.digest()).toLowerCase();
        String sha512Digest = DatatypeConverter.printHexBinary((byte[])sha512.digest()).toLowerCase();
        hexDigests.put(DefaultHashAlgorithms.MD5.getName(), md5Digest);
        hexDigests.put(DefaultHashAlgorithms.SHA_1.getName(), sha1Digest);
        hexDigests.put(DefaultHashAlgorithms.SHA_256.getName(), sha256Digest);
        hexDigests.put(DefaultHashAlgorithms.SHA_384.getName(), sha384Digest);
        hexDigests.put(DefaultHashAlgorithms.SHA_512.getName(), sha512Digest);
        if (generateAddAlgo) {
            String extraAlgoDigest = DatatypeConverter.printHexBinary((byte[])additionalAlgo.digest()).toLowerCase();
            hexDigests.put(additionalAlgorithm, extraAlgoDigest);
        }
        if (generateCsAlgo) {
            String extraChecksumDigest = DatatypeConverter.printHexBinary((byte[])checksumAlgo.digest()).toLowerCase();
            hexDigests.put(checksumAlgorithm, extraChecksumDigest);
        }
        logFileHashStore.debug((Object)("Object has been written to tmpFile: " + tmpFile.getName() + ". To be moved to: " + sha256Digest));
        return hexDigests;
    }

    protected void move(File source, File target, String entity) throws IOException, SecurityException, AtomicMoveNotSupportedException, FileAlreadyExistsException {
        logFileHashStore.debug((Object)("Moving " + entity + ", from source: " + source + ", to target: " + target));
        FileHashStoreUtility.ensureNotNull(entity, "entity");
        FileHashStoreUtility.checkForNotEmptyAndValidString(entity, "entity");
        if (entity.equals("object") && target.exists()) {
            String errMsg = "File already exists for target: " + target;
            logFileHashStore.warn((Object)errMsg);
            return;
        }
        FileHashStoreUtility.createParentDirectories(target.toPath());
        Path sourceFilePath = source.toPath();
        Path targetFilePath = target.toPath();
        try {
            Files.move(sourceFilePath, targetFilePath, StandardCopyOption.ATOMIC_MOVE);
            logFileHashStore.debug((Object)("File moved from: " + sourceFilePath + ", to: " + targetFilePath));
        }
        catch (FileAlreadyExistsException faee) {
            logFileHashStore.debug((Object)("File already exists, skipping request to move object. Source: " + source + ". Target: " + target));
        }
        catch (AtomicMoveNotSupportedException amnse) {
            logFileHashStore.error((Object)("StandardCopyOption.ATOMIC_MOVE failed. AtomicMove is not supported across file systems. Source: " + source + ". Target: " + target));
            throw amnse;
        }
        catch (IOException ioe) {
            logFileHashStore.error((Object)("Unable to move file. Source: " + source + ". Target: " + target));
            throw ioe;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void deleteObjectByCid(String cid) throws IOException, NoSuchAlgorithmException, InterruptedException {
        logFileHashStore.debug((Object)("Called to delete data object with cid: " + cid));
        Path absCidRefsPath = this.getHashStoreRefsPath(cid, HashStoreIdTypes.cid);
        String objRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, cid);
        Path expectedRealPath = this.OBJECT_STORE_DIRECTORY.resolve(objRelativePath);
        try {
            FileHashStore.synchronizeObjectLockedCids(cid);
            if (Files.exists(absCidRefsPath, new LinkOption[0])) {
                String warnMsg = "cid refs file still contains references, skipping deletion.";
                logFileHashStore.warn((Object)warnMsg);
            } else {
                if (Files.exists(expectedRealPath, new LinkOption[0])) {
                    Files.delete(expectedRealPath);
                }
                String debugMsg = "Object deleted at" + expectedRealPath;
                logFileHashStore.debug((Object)debugMsg);
            }
        }
        finally {
            FileHashStore.releaseObjectLockedCids(cid);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void storeHashStoreRefsFiles(String pid, String cid) throws NoSuchAlgorithmException, IOException, InterruptedException {
        try {
            Path absCidRefsPath;
            Path absPidRefsPath;
            block13: {
                FileHashStore.synchronizeObjectLockedCids(cid);
                FileHashStore.synchronizeReferenceLockedPids(pid);
                absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
                absCidRefsPath = this.getHashStoreRefsPath(cid, HashStoreIdTypes.cid);
                if (Files.exists(absPidRefsPath, new LinkOption[0]) && Files.exists(absCidRefsPath, new LinkOption[0])) {
                    String errMsg = "Object with cid: " + cid + " already exists and is tagged with pid: " + pid;
                    try {
                        this.verifyHashStoreRefsFiles(pid, cid, absPidRefsPath, absCidRefsPath);
                        logFileHashStore.error((Object)errMsg);
                        throw new HashStoreRefsAlreadyExistException(errMsg);
                    }
                    catch (Exception e) {
                        String revMsg = errMsg + " . " + e.getMessage();
                        logFileHashStore.error((Object)revMsg);
                        throw new HashStoreRefsAlreadyExistException(revMsg);
                    }
                }
                if (Files.exists(absPidRefsPath, new LinkOption[0]) && !Files.exists(absCidRefsPath, new LinkOption[0])) {
                    String errMsg = "Pid refs file already exists for pid: " + pid + ", and the associated cid refs file contains the pid. A pid can only reference one cid.";
                    logFileHashStore.error((Object)errMsg);
                    throw new PidRefsFileExistsException(errMsg);
                }
                if (Files.exists(absPidRefsPath, new LinkOption[0]) || !Files.exists(absCidRefsPath, new LinkOption[0])) break block13;
                File pidRefsTmpFile = this.writeRefsFile(cid, HashStoreIdTypes.pid.name());
                File absPathPidRefsFile = absPidRefsPath.toFile();
                this.move(pidRefsTmpFile, absPathPidRefsFile, "refs");
                if (!this.isStringInRefsFile(pid, absCidRefsPath)) {
                    this.updateRefsFile(pid, absCidRefsPath, HashStoreRefUpdateTypes.add);
                }
                this.verifyHashStoreRefsFiles(pid, cid, absPidRefsPath, absCidRefsPath);
                logFileHashStore.info((Object)("Object with cid: " + cid + " has been updated and tagged successfully with pid: " + pid));
                return;
            }
            try {
                File pidRefsTmpFile = this.writeRefsFile(cid, HashStoreIdTypes.pid.name());
                File cidRefsTmpFile = this.writeRefsFile(pid, HashStoreIdTypes.cid.name());
                File absPathPidRefsFile = absPidRefsPath.toFile();
                File absPathCidRefsFile = absCidRefsPath.toFile();
                this.move(pidRefsTmpFile, absPathPidRefsFile, "refs");
                this.move(cidRefsTmpFile, absPathCidRefsFile, "refs");
                this.verifyHashStoreRefsFiles(pid, cid, absPidRefsPath, absCidRefsPath);
                logFileHashStore.info((Object)("Object with cid: " + cid + " has been tagged successfully with pid: " + pid));
            }
            catch (HashStoreRefsAlreadyExistException | PidRefsFileExistsException hse) {
                throw hse;
            }
            catch (Exception e) {
                this.unTagObject(pid, cid);
                throw e;
            }
        }
        finally {
            FileHashStore.releaseObjectLockedCids(cid);
            FileHashStore.releaseReferenceLockedPids(pid);
        }
    }

    protected void unTagObject(String pid, String cid) throws InterruptedException, NoSuchAlgorithmException, IOException {
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        FileHashStoreUtility.ensureNotNull(cid, "cid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(cid, "cid");
        ArrayList<Path> deleteList = new ArrayList<Path>();
        if (!referenceLockedPids.contains(pid)) {
            String errMsg = "Cannot untag pid that is not currently locked";
            logFileHashStore.error((Object)errMsg);
            throw new IdentifierNotLockedException(errMsg);
        }
        try {
            ObjectInfo objInfo = this.findObject(pid);
            String cidToCheck = objInfo.cid();
            FileHashStore.validateAndCheckCidLock(pid, cid, cidToCheck);
            Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
            FileHashStore.markPidRefsFileForDeletion(pid, deleteList, absPidRefsPath);
            this.removePidAndHandleCidDeletion(pid, cid, deleteList);
            FileHashStore.deleteMarkedFiles(pid, cid, deleteList);
            logFileHashStore.info((Object)("Untagged pid: " + pid + " with cid: " + cid));
        }
        catch (OrphanPidRefsFileException oprfe) {
            Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
            String cidToCheck = new String(Files.readAllBytes(absPidRefsPath));
            FileHashStore.validateAndCheckCidLock(pid, cid, cidToCheck);
            FileHashStore.markPidRefsFileForDeletion(pid, deleteList, absPidRefsPath);
            FileHashStore.deleteMarkedFiles(pid, cid, deleteList);
            String warnMsg = "Cid refs file does not exist for pid: " + pid + ". Deleted orphan pid refs file.";
            logFileHashStore.warn((Object)warnMsg);
        }
        catch (OrphanRefsFilesException orfe) {
            Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
            String cidToCheck = new String(Files.readAllBytes(absPidRefsPath));
            FileHashStore.validateAndCheckCidLock(pid, cid, cidToCheck);
            FileHashStore.markPidRefsFileForDeletion(pid, deleteList, absPidRefsPath);
            this.removePidAndHandleCidDeletion(pid, cid, deleteList);
            FileHashStore.deleteMarkedFiles(pid, cid, deleteList);
            String warnMsg = "Object with cid: " + cidToCheck + " does not exist, but pid and cid reference file found for pid: " + pid + ". Deleted pid and cid ref files.";
            logFileHashStore.warn((Object)warnMsg);
        }
        catch (PidNotFoundInCidRefsFileException pnficrfe) {
            Path absPidRefsPath = this.getHashStoreRefsPath(pid, HashStoreIdTypes.pid);
            String cidToCheck = new String(Files.readAllBytes(absPidRefsPath));
            FileHashStore.validateAndCheckCidLock(pid, cid, cidToCheck);
            FileHashStore.markPidRefsFileForDeletion(pid, deleteList, absPidRefsPath);
            FileHashStore.deleteMarkedFiles(pid, cid, deleteList);
            String warnMsg = "Pid not found in expected cid refs file for pid: " + pid + ". Deleted orphan pid refs file.";
            logFileHashStore.warn((Object)warnMsg);
        }
        catch (PidRefsFileNotFoundException prfnfe) {
            if (!objectLockedCids.contains(cid)) {
                String errMsg = "Cannot untag cid that is not currently locked";
                logFileHashStore.error((Object)errMsg);
                throw new IdentifierNotLockedException(errMsg);
            }
            this.removePidAndHandleCidDeletion(pid, cid, deleteList);
            FileHashStore.deleteMarkedFiles(pid, cid, deleteList);
            String errMsg = "Pid refs file not found, removed pid from cid refs file for cid: " + cid;
            logFileHashStore.warn((Object)errMsg);
        }
    }

    private static void validateAndCheckCidLock(String pid, String cid, String cidToCheck) {
        FileHashStoreUtility.ensureNotNull(cidToCheck, "cidToCheck");
        FileHashStoreUtility.checkForNotEmptyAndValidString(cidToCheck, "cidToCheck");
        if (!cid.equals(cidToCheck)) {
            String errMsg = "Cid retrieved: " + cidToCheck + " does not match untag request for cid: " + cid + " and pid: " + pid;
            logFileHashStore.error((Object)errMsg);
            throw new IllegalArgumentException(errMsg);
        }
        if (!objectLockedCids.contains(cid)) {
            String errMsg = "Cannot untag cid: " + cid + " that is not currently locked (pid: " + pid + ")";
            logFileHashStore.error((Object)errMsg);
            throw new IdentifierNotLockedException(errMsg);
        }
    }

    private void removePidAndHandleCidDeletion(String pid, String cid, Collection<Path> deleteList) {
        Path absCidRefsPath = null;
        try {
            absCidRefsPath = this.getHashStoreRefsPath(cid, HashStoreIdTypes.cid);
            this.updateRefsFile(pid, absCidRefsPath, HashStoreRefUpdateTypes.remove);
            if (Files.size(absCidRefsPath) == 0L) {
                deleteList.add(FileHashStoreUtility.renamePathForDeletion(absCidRefsPath));
            } else {
                String infoMsg = "Cid referenced by pid: " + pid + " is not empty (refs exist for cid). Skipping object deletion.";
                logFileHashStore.info((Object)infoMsg);
            }
        }
        catch (Exception e) {
            logFileHashStore.error((Object)("Unable to remove pid: " + pid + " from cid refs file: " + absCidRefsPath + ". " + e.getMessage()));
        }
    }

    private static void deleteMarkedFiles(String pid, String cid, Collection<Path> deleteList) {
        try {
            FileHashStoreUtility.deleteListItems(deleteList);
        }
        catch (Exception e) {
            logFileHashStore.error((Object)("Unable to delete list of refs files marked for deletion for request with pid: " + pid + " and cid: " + cid + ". " + e.getMessage()));
        }
    }

    private static void markPidRefsFileForDeletion(String pid, Collection<Path> deleteList, Path absPidRefsPath) {
        try {
            deleteList.add(FileHashStoreUtility.renamePathForDeletion(absPidRefsPath));
        }
        catch (Exception e) {
            logFileHashStore.error((Object)("Unable to delete pid refs file: " + absPidRefsPath + " for pid: " + pid + ". " + e.getMessage()));
        }
    }

    protected void verifyHashStoreRefsFiles(String pid, String cid, Path absPidRefsPath, Path absCidRefsPath) throws FileNotFoundException, CidNotFoundInPidRefsFileException, PidNotFoundInCidRefsFileException, IOException {
        if (!Files.exists(absCidRefsPath, new LinkOption[0])) {
            String errMsg = "Cid refs file is missing: " + absCidRefsPath + " for pid: " + pid;
            logFileHashStore.error((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        if (!Files.exists(absPidRefsPath, new LinkOption[0])) {
            String errMsg = "Pid refs file is missing: " + absPidRefsPath + " for cid: " + cid;
            logFileHashStore.error((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        try {
            String cidRead = new String(Files.readAllBytes(absPidRefsPath));
            if (!cidRead.equals(cid)) {
                String errMsg = "Unexpected cid: " + cidRead + " found in pid refs file: " + absPidRefsPath + ". Expected cid: " + cid;
                logFileHashStore.error((Object)errMsg);
                throw new CidNotFoundInPidRefsFileException(errMsg);
            }
            if (!this.isStringInRefsFile(pid, absCidRefsPath)) {
                String errMsg = "Missing expected pid: " + pid + " in cid refs file: " + absCidRefsPath;
                logFileHashStore.error((Object)errMsg);
                throw new PidNotFoundInCidRefsFileException(errMsg);
            }
        }
        catch (IOException ioe) {
            logFileHashStore.error((Object)ioe.getMessage());
            throw ioe;
        }
    }

    protected File writeRefsFile(String ref, String refType) throws IOException {
        File file;
        File cidRefsTmpFile = FileHashStoreUtility.generateTmpFile("tmp", this.REFS_TMP_FILE_DIRECTORY);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(Files.newOutputStream(cidRefsTmpFile.toPath(), new OpenOption[0]), StandardCharsets.UTF_8));
        try {
            writer.write(ref);
            writer.close();
            logFileHashStore.debug((Object)(refType + " refs file written for: " + ref));
            file = cidRefsTmpFile;
        }
        catch (Throwable throwable) {
            try {
                try {
                    writer.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ioe) {
                String errMsg = "Unable to write refs file for ref: " + refType + " IOException: " + ioe.getMessage();
                logFileHashStore.error((Object)errMsg);
                throw new IOException(errMsg);
            }
        }
        writer.close();
        return file;
    }

    protected boolean isStringInRefsFile(String ref, Path absRefsPath) throws IOException {
        List<String> lines = Files.readAllLines(absRefsPath);
        boolean refFoundInCidRefFiles = false;
        for (String line : lines) {
            if (!line.equals(ref)) continue;
            refFoundInCidRefFiles = true;
            break;
        }
        return refFoundInCidRefFiles;
    }

    protected void updateRefsFile(String ref, Path absRefsPath, HashStoreRefUpdateTypes updateType) throws IOException {
        File tmpFile = FileHashStoreUtility.generateTmpFile("tmp", this.REFS_TMP_FILE_DIRECTORY);
        Path tmpFilePath = tmpFile.toPath();
        try (FileChannel channel = FileChannel.open(absRefsPath, StandardOpenOption.READ, StandardOpenOption.WRITE);
             FileLock ignored = channel.lock();){
            ArrayList<String> lines = new ArrayList<String>(Files.readAllLines(absRefsPath));
            if (updateType.equals((Object)HashStoreRefUpdateTypes.add)) {
                if (!lines.contains(ref)) {
                    lines.add(ref);
                    Files.write(tmpFilePath, lines, StandardOpenOption.WRITE);
                    this.move(tmpFile, absRefsPath.toFile(), "refs");
                    logFileHashStore.debug((Object)("Ref: " + ref + " has been added to refs file: " + absRefsPath));
                } else {
                    logFileHashStore.debug((Object)("Ref: " + ref + " already exists in refs file: " + absRefsPath));
                }
            } else if (updateType.equals((Object)HashStoreRefUpdateTypes.remove)) {
                lines.remove(ref);
                Files.write(tmpFilePath, lines, StandardOpenOption.WRITE);
                this.move(tmpFile, absRefsPath.toFile(), "refs");
                logFileHashStore.debug((Object)("Ref: " + ref + " has been removed from refs file: " + absRefsPath));
            }
        }
        catch (IOException ioe) {
            logFileHashStore.error((Object)ioe.getMessage());
            throw ioe;
        }
    }

    protected String putMetadata(InputStream metadata, String pid, String formatId) throws NoSuchAlgorithmException, IOException {
        String checkedFormatId;
        logFileHashStore.debug((Object)("Writing metadata for pid: " + pid + " , with metadata namespace: " + formatId));
        FileHashStoreUtility.ensureNotNull(metadata, "metadata");
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        if (formatId == null) {
            checkedFormatId = this.DEFAULT_METADATA_NAMESPACE;
        } else {
            FileHashStoreUtility.checkForNotEmptyAndValidString(formatId, "formatId");
            checkedFormatId = formatId;
        }
        Path pathToStoredMetadata = this.getHashStoreMetadataPath(pid, checkedFormatId);
        File tmpMetadataFile = FileHashStoreUtility.generateTmpFile("tmp", this.METADATA_TMP_FILE_DIRECTORY);
        boolean tmpMetadataWritten = this.writeToTmpMetadataFile(tmpMetadataFile, metadata);
        if (tmpMetadataWritten) {
            logFileHashStore.debug((Object)("Tmp metadata file has been written, moving to permanent location: " + pathToStoredMetadata));
            File permMetadataFile = pathToStoredMetadata.toFile();
            this.move(tmpMetadataFile, permMetadataFile, "metadata");
        }
        logFileHashStore.debug((Object)("Metadata moved successfully, permanent address: " + pathToStoredMetadata));
        return pathToStoredMetadata.toString();
    }

    protected boolean writeToTmpMetadataFile(File tmpFile, InputStream metadataStream) throws IOException, FileNotFoundException {
        FileOutputStream os = new FileOutputStream(tmpFile);
        try {
            int bytesRead;
            byte[] buffer = new byte[8192];
            while ((bytesRead = metadataStream.read(buffer)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
            boolean bl = true;
            return bl;
        }
        catch (IOException ioe) {
            logFileHashStore.error((Object)ioe.getMessage());
            throw ioe;
        }
        finally {
            os.flush();
            os.close();
        }
    }

    protected Path getHashStoreDataObjectPath(String abpId) throws NoSuchAlgorithmException, IOException {
        String hashedId = FileHashStoreUtility.getPidHexDigest(abpId, this.OBJECT_STORE_ALGORITHM);
        String pidRefsFileRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, hashedId);
        Path pathToPidRefsFile = this.REFS_PID_FILE_DIRECTORY.resolve(pidRefsFileRelativePath);
        if (!Files.exists(pathToPidRefsFile, new LinkOption[0])) {
            String errMsg = "Pid Refs file does not exist for pid: " + abpId + " with object address: " + pathToPidRefsFile + ". Cannot retrieve cid.";
            logFileHashStore.warn((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        String objectCid = new String(Files.readAllBytes(pathToPidRefsFile));
        String objRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, objectCid);
        return this.OBJECT_STORE_DIRECTORY.resolve(objRelativePath);
    }

    protected Path getHashStoreMetadataPath(String abpId, String formatId) throws NoSuchAlgorithmException {
        String hashedId = FileHashStoreUtility.getPidHexDigest(abpId, this.OBJECT_STORE_ALGORITHM);
        String pidMetadataDirRelPath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, hashedId);
        String metadataDocHash = FileHashStoreUtility.getPidHexDigest(abpId + formatId, this.OBJECT_STORE_ALGORITHM);
        return this.METADATA_STORE_DIRECTORY.resolve(pidMetadataDirRelPath).resolve(metadataDocHash);
    }

    protected InputStream getHashStoreMetadataInputStream(String pid, String formatId) throws NoSuchAlgorithmException, IOException, FileNotFoundException {
        Path metadataCidPath = this.getHashStoreMetadataPath(pid, formatId);
        if (!Files.exists(metadataCidPath, new LinkOption[0])) {
            String errMsg = "Metadata does not exist for pid: " + pid + " with formatId: " + formatId + ". Metadata address: " + metadataCidPath;
            logFileHashStore.warn((Object)errMsg);
            throw new FileNotFoundException(errMsg);
        }
        try {
            InputStream metadataCidInputStream = Files.newInputStream(metadataCidPath, new OpenOption[0]);
            logFileHashStore.info((Object)("Retrieved metadata for pid: " + pid + " with formatId: " + formatId));
            return metadataCidInputStream;
        }
        catch (IOException ioe) {
            String errMsg = "Unexpected error when creating InputStream for pid: " + pid + " with formatId: " + formatId + ". IOException: " + ioe.getMessage();
            logFileHashStore.error((Object)errMsg);
            throw new IOException(errMsg);
        }
    }

    protected Path getHashStoreRefsPath(String abpcId, HashStoreIdTypes refType) throws NoSuchAlgorithmException {
        return switch (refType) {
            case HashStoreIdTypes.pid -> {
                String hashedId = FileHashStoreUtility.getPidHexDigest(abpcId, this.OBJECT_STORE_ALGORITHM);
                String pidRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, hashedId);
                yield this.REFS_PID_FILE_DIRECTORY.resolve(pidRelativePath);
            }
            case HashStoreIdTypes.cid -> {
                String cidRelativePath = FileHashStoreUtility.getHierarchicalPathString(this.DIRECTORY_DEPTH, this.DIRECTORY_WIDTH, abpcId);
                yield this.REFS_CID_FILE_DIRECTORY.resolve(cidRelativePath);
            }
            default -> throw new IllegalArgumentException("Ref type must be a type of HashStoreIdTypes (pid or cid)");
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void synchronizeObjectLockedPids(String pid) throws InterruptedException {
        Collection<String> collection = objectLockedPids;
        synchronized (collection) {
            while (objectLockedPids.contains(pid)) {
                try {
                    objectLockedPids.wait(1000L);
                }
                catch (InterruptedException ie) {
                    String errMsg = "Synchronization has been interrupted while trying to sync pid: " + pid;
                    logFileHashStore.warn((Object)errMsg);
                    throw new InterruptedException(errMsg);
                }
            }
            logFileHashStore.debug((Object)("Synchronizing objectLockedPids for pid: " + pid));
            objectLockedPids.add(pid);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void releaseObjectLockedPids(String pid) {
        Collection<String> collection = objectLockedPids;
        synchronized (collection) {
            logFileHashStore.debug((Object)("Releasing objectLockedPids for pid: " + pid));
            objectLockedPids.remove(pid);
            objectLockedPids.notify();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void synchronizeMetadataLockedDocIds(String metadataDocId) throws InterruptedException {
        Collection<String> collection = metadataLockedDocIds;
        synchronized (collection) {
            while (metadataLockedDocIds.contains(metadataDocId)) {
                try {
                    metadataLockedDocIds.wait(1000L);
                }
                catch (InterruptedException ie) {
                    String errMsg = "Synchronization has been interrupted while trying to sync metadata doc: " + metadataDocId;
                    logFileHashStore.error((Object)errMsg);
                    throw new InterruptedException(errMsg);
                }
            }
            logFileHashStore.debug((Object)("Synchronizing metadataLockedDocIds for metadata doc: " + metadataDocId));
            metadataLockedDocIds.add(metadataDocId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void releaseMetadataLockedDocIds(String metadataDocId) {
        Collection<String> collection = metadataLockedDocIds;
        synchronized (collection) {
            logFileHashStore.debug((Object)("Releasing metadataLockedDocIds for metadata doc: " + metadataDocId));
            metadataLockedDocIds.remove(metadataDocId);
            metadataLockedDocIds.notify();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void synchronizeObjectLockedCids(String cid) throws InterruptedException {
        Collection<String> collection = objectLockedCids;
        synchronized (collection) {
            while (objectLockedCids.contains(cid)) {
                try {
                    objectLockedCids.wait(1000L);
                }
                catch (InterruptedException ie) {
                    String errMsg = "Synchronization has been interrupted while trying to sync cid: " + cid;
                    logFileHashStore.error((Object)errMsg);
                    throw new InterruptedException(errMsg);
                }
            }
            logFileHashStore.debug((Object)("Synchronizing objectLockedCids for cid: " + cid));
            objectLockedCids.add(cid);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void releaseObjectLockedCids(String cid) {
        Collection<String> collection = objectLockedCids;
        synchronized (collection) {
            logFileHashStore.debug((Object)("Releasing objectLockedCids for cid: " + cid));
            objectLockedCids.remove(cid);
            objectLockedCids.notify();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void synchronizeReferenceLockedPids(String pid) throws InterruptedException {
        Collection<String> collection = referenceLockedPids;
        synchronized (collection) {
            while (referenceLockedPids.contains(pid)) {
                try {
                    referenceLockedPids.wait(1000L);
                }
                catch (InterruptedException ie) {
                    String errMsg = "Synchronization has been interrupted while trying to sync pid: " + pid;
                    logFileHashStore.error((Object)errMsg);
                    throw new InterruptedException(errMsg);
                }
            }
            logFileHashStore.debug((Object)("Synchronizing referenceLockedPids for pid: " + pid));
            referenceLockedPids.add(pid);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void releaseReferenceLockedPids(String pid) {
        Collection<String> collection = referenceLockedPids;
        synchronized (collection) {
            logFileHashStore.debug((Object)("Releasing referenceLockedPids for pid: " + pid));
            referenceLockedPids.remove(pid);
            referenceLockedPids.notify();
        }
    }

    protected static enum HashStoreProperties {
        storePath,
        storeDepth,
        storeWidth,
        storeAlgorithm,
        storeMetadataNamespace;

    }

    protected record ObjectInfo(String cid, String cidObjectPath, String cidRefsPath, String pidRefsPath, String sysmetaPath) {
    }

    protected static enum HashStoreIdTypes {
        cid,
        pid;

    }

    protected static enum HashStoreRefUpdateTypes {
        add,
        remove;

    }

    protected static enum DefaultHashAlgorithms {
        MD5("MD5"),
        SHA_1("SHA-1"),
        SHA_256("SHA-256"),
        SHA_384("SHA-384"),
        SHA_512("SHA-512");

        final String algoName;

        private DefaultHashAlgorithms(String algo) {
            this.algoName = algo;
        }

        public String getName() {
            return this.algoName;
        }
    }
}

