/*
 * Decompiled with CFR 0.152.
 */
package org.dataone.hashstore.filehashstore;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.stream.Stream;
import javax.xml.bind.DatatypeConverter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class FileHashStoreUtility {
    private static final Log log = LogFactory.getLog(FileHashStoreUtility.class);

    public static void ensureNotNull(Object object, String argument) throws IllegalArgumentException {
        if (object == null) {
            StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
            String msg = "Calling Method: " + stackTraceElements[2].getMethodName() + "()'s argument: " + argument + " cannot be null.";
            throw new IllegalArgumentException(msg);
        }
    }

    public static String calculateHexDigest(InputStream dataStream, String algorithm) throws IOException, NoSuchAlgorithmException {
        MessageDigest mdObject = MessageDigest.getInstance(algorithm);
        try (InputStream inputStream = dataStream;){
            int bytesRead;
            byte[] buffer = new byte[8192];
            while ((bytesRead = dataStream.read(buffer)) != -1) {
                mdObject.update(buffer, 0, bytesRead);
            }
        }
        catch (IOException ioe) {
            String errMsg = "Unexpected IOException encountered: " + ioe.getMessage();
            throw new IOException(errMsg);
        }
        return DatatypeConverter.printHexBinary((byte[])mdObject.digest()).toLowerCase();
    }

    public static String getPidHexDigest(String pid, String algorithm) throws NoSuchAlgorithmException, IllegalArgumentException {
        FileHashStoreUtility.ensureNotNull(pid, "pid");
        FileHashStoreUtility.checkForNotEmptyAndValidString(pid, "pid");
        FileHashStoreUtility.ensureNotNull(algorithm, "algorithm");
        FileHashStoreUtility.checkForNotEmptyAndValidString(algorithm, "algorithm");
        MessageDigest stringMessageDigest = MessageDigest.getInstance(algorithm);
        byte[] bytes = pid.getBytes(StandardCharsets.UTF_8);
        stringMessageDigest.update(bytes);
        return DatatypeConverter.printHexBinary((byte[])stringMessageDigest.digest()).toLowerCase();
    }

    public static void createParentDirectories(Path desiredPath) throws IOException {
        File desiredPathParentDirs = new File(desiredPath.toFile().getParent());
        if (!desiredPathParentDirs.exists()) {
            Path destinationDirectoryPath = desiredPathParentDirs.toPath();
            try {
                Set<PosixFilePermission> perms = PosixFilePermissions.fromString("rwxr-x---");
                FileAttribute<Set<PosixFilePermission>> attr = PosixFilePermissions.asFileAttribute(perms);
                Files.createDirectories(destinationDirectoryPath, attr);
            }
            catch (FileAlreadyExistsException faee) {
                log.warn((Object)("Directory already exists at: " + destinationDirectoryPath + " - Skipping directory creation"));
            }
        }
    }

    public static boolean dirContainsFiles(Path directory) throws IOException {
        try (Stream<Path> stream = Files.list(directory);){
            boolean bl = stream.findFirst().isPresent();
            return bl;
        }
    }

    public static List<Path> getFilesFromDir(Path directory) throws IOException {
        ArrayList<Path> filePaths = new ArrayList<Path>();
        if (Files.isDirectory(directory, new LinkOption[0]) && FileHashStoreUtility.dirContainsFiles(directory)) {
            try (Stream<Path> stream = Files.walk(directory, new FileVisitOption[0]);){
                stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(filePaths::add);
            }
        }
        return filePaths;
    }

    public static Path renamePathForDeletion(Path pathToRename) throws IOException {
        FileHashStoreUtility.ensureNotNull(pathToRename, "pathToRename");
        if (!Files.exists(pathToRename, new LinkOption[0])) {
            String errMsg = "Given path to file: " + pathToRename + " does not exist.";
            throw new FileNotFoundException(errMsg);
        }
        Path parentPath = pathToRename.getParent();
        Path fileName = pathToRename.getFileName();
        String newFileName = fileName.toString() + "_delete";
        Path deletePath = parentPath.resolve(newFileName);
        Files.move(pathToRename, deletePath, StandardCopyOption.ATOMIC_MOVE);
        return deletePath;
    }

    public static void renamePathForRestoration(Path pathToRename) throws IOException {
        FileHashStoreUtility.ensureNotNull(pathToRename, "pathToRename");
        if (!Files.exists(pathToRename, new LinkOption[0])) {
            String errMsg = "Given path to file: " + pathToRename + " does not exist.";
            throw new FileNotFoundException(errMsg);
        }
        Path parentPath = pathToRename.getParent();
        Path fileName = pathToRename.getFileName();
        String newFileName = fileName.toString().replace("_delete", "");
        Path restorePath = parentPath.resolve(newFileName);
        Files.move(pathToRename, restorePath, StandardCopyOption.ATOMIC_MOVE);
    }

    public static void deleteListItems(Collection<Path> deleteList) {
        FileHashStoreUtility.ensureNotNull(deleteList, "deleteList");
        if (!deleteList.isEmpty()) {
            for (Path deleteItem : deleteList) {
                if (!Files.exists(deleteItem, new LinkOption[0])) continue;
                try {
                    Files.delete(deleteItem);
                }
                catch (Exception ge) {
                    String warnMsg = "Attempted to delete metadata document: " + deleteItem + " but failed. Additional Details: " + ge.getMessage();
                    log.warn((Object)warnMsg);
                }
            }
        }
    }

    public static void checkForNotEmptyAndValidString(String string, String argument) throws IllegalArgumentException {
        FileHashStoreUtility.ensureNotNull(string, "string");
        if (string.isBlank()) {
            StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
            String msg = "Calling Method: " + stackTraceElements[2].getMethodName() + "()'s argument: " + argument + " cannot be empty, contain empty white spaces, tabs or newlines.";
            throw new IllegalArgumentException(msg);
        }
        if (!FileHashStoreUtility.isValidString(string)) {
            StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
            String msg = "Calling Method: " + stackTraceElements[2].getMethodName() + "()'s argument: " + argument + " contains empty white spaces, tabs or newlines.";
            throw new IllegalArgumentException(msg);
        }
    }

    public static boolean isValidString(String string) {
        boolean valid = true;
        for (int i = 0; i < string.length(); ++i) {
            char ch = string.charAt(i);
            if (!Character.isWhitespace(ch)) continue;
            valid = false;
            break;
        }
        return valid;
    }

    public static void checkPositive(long longInt) throws IllegalArgumentException {
        if (longInt <= 0L) {
            StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
            String msg = "Calling Method: " + stackTraceElements[2].getMethodName() + "(): given objSize/long/runnableMethod/etc. object cannot be <= 0 ";
            throw new IllegalArgumentException(msg);
        }
    }

    public static String getHierarchicalPathString(int depth, int width, String digest) {
        ArrayList<String> tokens = new ArrayList<String>();
        int digestLength = digest.length();
        for (int i = 0; i < depth; ++i) {
            int start = i * width;
            int end = Math.min((i + 1) * width, digestLength);
            tokens.add(digest.substring(start, end));
        }
        if (depth * width < digestLength) {
            tokens.add(digest.substring(depth * width));
        }
        ArrayList<String> stringArray = new ArrayList<String>();
        for (String str : tokens) {
            if (str.trim().isEmpty()) continue;
            stringArray.add(str);
        }
        return String.join((CharSequence)"/", stringArray);
    }

    public static File generateTmpFile(String prefix, Path directory) throws IOException, SecurityException {
        Random rand = new Random();
        int randomNumber = rand.nextInt(1000000);
        String newPrefix = prefix + "-" + System.currentTimeMillis() + randomNumber;
        Path newTmpPath = Files.createTempFile(directory, newPrefix, null, new FileAttribute[0]);
        File newTmpFile = newTmpPath.toFile();
        HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>();
        permissions.add(PosixFilePermission.OWNER_READ);
        permissions.add(PosixFilePermission.OWNER_WRITE);
        permissions.add(PosixFilePermission.GROUP_READ);
        Files.setPosixFilePermissions(newTmpPath, permissions);
        newTmpFile.deleteOnExit();
        return newTmpFile;
    }

    public static void checkObjectEquality(String nameValue, Object suppliedValue, Object existingValue) {
        if (!Objects.equals(suppliedValue, existingValue)) {
            String errMsg = "FileHashStore.checkConfigurationEquality() - Mismatch in " + nameValue + ": " + suppliedValue + " does not match the existing configuration value: " + existingValue;
            throw new IllegalArgumentException(errMsg);
        }
    }
}

